ShapeDrawer = class()

function ShapeDrawer:init()
end

function ShapeDrawer:drawShape(body, strokeColor, strokeWidthNum)
    pushStyle()
    pushMatrix()
    translate(body.x, body.y)
    rotate(body.angle)
    stroke(strokeColor or stroke())
    strokeWidth(strokeWidthNum or strokeWidth())
    if body.radius then
        line(0,0,body.radius-3,0)            
        ellipse(0,0,body.radius*2)
    elseif body.shapeType == CHAIN or body.shapeType == EDGE then
        local points = body.points
        for j = 1,#points-1 do
            a = points[j]
            b = points[j+1]
            line(a.x, a.y, b.x, b.y)
        end  
    else 
        local points = body.points
        for j = 1,#points do
            a = points[j]
            b = points[(j % #points)+1]
            line(a.x, a.y, b.x, b.y)
        end
    end    
    popMatrix()
    popStyle()
end

function ShapeDrawer:drawJoints(joints)
    for k,joint in pairs(joints) do
        if joint and joint.anchorA and joint.anchorB then
            local a = joint.anchorA
            local b = joint.anchorB
            line(a.x,a.y,b.x,b.y)
        end
    end
end

function ShapeDrawer:drawBodies(bodies)
    for i,body in ipairs(bodies) do
        if body and body.x and body.y then
            pushMatrix()
            translate(body.x, body.y)
            rotate(body.angle)
            
            if body.type == STATIC then
                stroke(255,255,255,255)
            elseif body.type == DYNAMIC then
                stroke(150,255,150,255)
            elseif body.type == KINEMATIC then
                stroke(150,150,255,255)
            end
            
            if body.shapeType == POLYGON then
                strokeWidth(3.0)
                local points = body.points
                for j = 1,#points do
                    a = points[j]
                    b = points[(j % #points)+1]
                    line(a.x, a.y, b.x, b.y)
                end
            elseif body.shapeType == CHAIN or body.shapeType == EDGE then
                strokeWidth(3.0)
                local points = body.points
                for j = 1,#points-1 do
                    a = points[j]
                    b = points[j+1]
                    line(a.x, a.y, b.x, b.y)
                end      
            elseif body.shapeType == CIRCLE then
                strokeWidth(3.0)
                line(0,0,body.radius-3,0)            
                ellipse(0,0,body.radius*2)
            end            
            popMatrix() 
        end
    end 
end

function ShapeDrawer:drawContacts(contacts)
    for k,v in pairs(contacts) do
        for m,n in ipairs(v.points) do
            ellipse(n.x, n.y, 10, 10)
        end
    end
end

function ShapeDrawer:drawTouchMap(touchMap)
    for k,v in pairs(touchMap) do
        local worldAnchor = v.body:getWorldPoint(v.anchor)
        local touchPoint = v.tp        
        line(touchPoint.x, touchPoint.y, worldAnchor.x, worldAnchor.y)
    end
end

function ShapeDrawer:drawThese(bodies, joints, contacts, touchMap)
    
    pushStyle()
    lineCapMode(ROUND)
    smooth()
    strokeWidth(5)
    stroke(128,0,128)
    
    self:drawTouchMap(touchMap)
    
    stroke(0,255,0,255)
    strokeWidth(5)
    
    self:drawJoints(joints)
    
    stroke(255,255,255,255)
    noFill()
    
    self:drawBodies(bodies)
    
    stroke(255, 0, 0, 255)
    fill(255, 0, 0, 255)
    
    self:drawContacts(contacts)
    
    popStyle()
end

function ShapeDrawer:cornersBounding(body)
    local ll, ur = body.position, body.position
    if body.shapeType == CIRCLE then 
        ll = body.position - vec2(body.radius, body.radius)
        ur = body.position + vec2(body.radius, body.radius)
    else
        for _, thisPoint in ipairs(body.points) do
            local worldPoint = body:getWorldPoint(thisPoint)
            if worldPoint.x <= ll.x then ll.x = worldPoint.x end
            if worldPoint.y <= ll.y then ll.y = worldPoint.y end
            if worldPoint.x >= ur.x then ur.x = worldPoint.x end
            if worldPoint.y >= ur.y then ur.y = worldPoint.y end
        end 
    end  
    local lr, ul = vec2(ur.x, ll.y), vec2(ll.x, ur.y)
    return ll, lr, ul, ur
end

function ShapeDrawer:drawAABBOf(body, drawColor)
    local ll, _, _, ur = self:cornersBounding(body)
    if ll and ur then
        local foundUpper, foundLower, foundRight, foundLeft, foundAABBBounds
        while not foundAABBBounds do
            local yUp, yDown, xLeft, xRight = 0, 0, 0, 0
            --find left edge
            while not foundLeft do
                local bodiesToLeft = physics.queryAABB(vec2(0,0), vec2(ll.x - xLeft, HEIGHT))
                if tableHas(bodiesToLeft, body) then
                    xLeft = xLeft + 1
                else
                    xLeft = xLeft - 1
                    foundLeft = true
                end
            end
            --find right edge
            while not foundRight do
                local bodiesToRight = physics.queryAABB(vec2(ur.x + xRight, 0), vec2(WIDTH, HEIGHT))
                if tableHas(bodiesToRight, body) then
                    xRight = xRight + 1
                else
                    xRight = xRight - 1
                    foundRight = true
                end
            end
            --find top
            while not foundUpper do
                local bodiesAbove = physics.queryAABB(vec2(0, ur.y + yUp), vec2(WIDTH, HEIGHT))
                if tableHas(bodiesAbove, body) then
                    yUp = yUp + 1
                else
                    yUp = yUp - 1
                    foundUpper = true
                end
            end
            --find bottom
            while not foundLower do
                local bodiesBelow = physics.queryAABB(vec2(0, 0), vec2(WIDTH, ll.y - yDown))
                if tableHas(bodiesBelow, body) then
                    yDown = yDown + 1
                else
                    yDown = yDown - 1
                    foundLower = true
                end
            end
            
            ur.x = ur.x + xRight
            ur.y = ur.y + yUp
            
            ll.x = ll.x - xLeft
            ll.y = ll.y - yDown
            
            foundAABBBounds = true
        end
        pushStyle()
        noStroke()
        rectMode(CORNER)
        drawColor = drawColor or color(233, 80, 223, 62)
        fill(drawColor)
        rect(ll.x, ll.y, ur.x - ll.x, ur.y - ll.y)
        popStyle()
    end
end

function ShapeDrawer:drawAllAABB(drawColor, bodiesToIgnore)
    local all = physics.queryAABB(vec2(0,0),vec2(WIDTH, HEIGHT))
    local rendered = {}
    for _, body in ipairs(all) do             
        if bodiesToIgnore and tableHas(bodiesToIgnore, body) then goto continue end
        if tableHas(rendered, body) then goto continue end
        self:drawAABBOf(body, drawColor)
        table.insert(rendered, body)
        ::continue::
    end
end

